# 📖 Complete Manual Installation Guide - Version 2.2

## Step-by-Step Instructions to Add Authentication & Security

---

## Table of Contents

1. [Pre-Installation Checklist](#pre-installation-checklist)
2. [Database Changes](#database-changes)
3. [Files to Create (9 files)](#files-to-create)
4. [Files to Modify (1 file)](#files-to-modify)
5. [Testing Your Installation](#testing-your-installation)
6. [Troubleshooting](#troubleshooting)

---

## Pre-Installation Checklist

### ⚠️ CRITICAL: Backup First!

```bash
# Backup your database
mysqldump -u username -p investor_db > backup_before_v2.2_$(date +%Y%m%d).sql

# Backup your files
cp -r investor-database investor-database-backup-$(date +%Y%m%d)
```

### Requirements

- [ ] Existing v2.1 system installed and working
- [ ] Database backup completed
- [ ] File backup completed
- [ ] Access to phpMyAdmin or MySQL command line
- [ ] FTP/SFTP access to upload files
- [ ] Text editor (Notepad++, VS Code, etc.)

---

## Database Changes

### Yes, 4 New Tables Required

Run the following SQL in your database. You can:
- Use phpMyAdmin (Import SQL file)
- Use MySQL command line
- Copy/paste into SQL tab

### SQL Script: add_authentication.sql

**Save this as:** `database/add_authentication.sql`

```sql
-- Authentication System for Investor Database
-- Version 2.2 - Security & Authentication
-- Run this after add_introducers.sql

USE investor_db;

-- Create Users Table
CREATE TABLE IF NOT EXISTS users (
    user_id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) UNIQUE NOT NULL,
    email VARCHAR(150) UNIQUE NOT NULL,
    password_hash VARCHAR(255) NOT NULL,

    -- Personal Information
    first_name VARCHAR(100) NOT NULL,
    last_name VARCHAR(100) NOT NULL,
    phone VARCHAR(20),

    -- User Role & Permissions
    role ENUM('Admin', 'Manager', 'User', 'Viewer') DEFAULT 'User',
    staff_id INT NULL,

    -- Account Status
    status ENUM('Active', 'Inactive', 'Suspended', 'Locked') DEFAULT 'Active',
    email_verified TINYINT(1) DEFAULT 0,

    -- Security
    failed_login_attempts INT DEFAULT 0,
    last_failed_login TIMESTAMP NULL,
    last_login TIMESTAMP NULL,
    last_password_change TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    password_reset_token VARCHAR(64) NULL,
    password_reset_expires TIMESTAMP NULL,

    -- Session Management
    remember_token VARCHAR(64) NULL,

    -- Timestamps
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    created_by INT NULL,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    updated_by INT NULL,

    FOREIGN KEY (staff_id) REFERENCES staff(staff_id) ON DELETE SET NULL,
    INDEX idx_username (username),
    INDEX idx_email (email),
    INDEX idx_role (role),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Create Sessions Table
CREATE TABLE IF NOT EXISTS user_sessions (
    session_id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    session_token VARCHAR(64) UNIQUE NOT NULL,
    ip_address VARCHAR(45),
    user_agent TEXT,
    expires_at TIMESTAMP NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    FOREIGN KEY (user_id) REFERENCES users(user_id) ON DELETE CASCADE,
    INDEX idx_session_token (session_token),
    INDEX idx_user_id (user_id),
    INDEX idx_expires (expires_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Create Activity Log Table
CREATE TABLE IF NOT EXISTS activity_log (
    log_id BIGINT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NULL,
    action VARCHAR(100) NOT NULL,
    entity_type VARCHAR(50) NULL,
    entity_id INT NULL,
    description TEXT,
    ip_address VARCHAR(45),
    user_agent TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    FOREIGN KEY (user_id) REFERENCES users(user_id) ON DELETE SET NULL,
    INDEX idx_user_id (user_id),
    INDEX idx_action (action),
    INDEX idx_entity (entity_type, entity_id),
    INDEX idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Create Login Attempts Table
CREATE TABLE IF NOT EXISTS login_attempts (
    attempt_id BIGINT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) NOT NULL,
    ip_address VARCHAR(45),
    success TINYINT(1) NOT NULL,
    failure_reason VARCHAR(100) NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    INDEX idx_username (username),
    INDEX idx_ip (ip_address),
    INDEX idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert default admin user
-- Default password: Admin@123 (MUST BE CHANGED IMMEDIATELY)
INSERT INTO users (
    username, email, password_hash, first_name, last_name,
    role, status, email_verified
) VALUES (
    'admin',
    'admin@oasis.local',
    '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi',
    'System',
    'Administrator',
    'Admin',
    'Active',
    1
);

-- Add sample manager user
-- Default password: Manager@123 (MUST BE CHANGED)
INSERT INTO users (
    username, email, password_hash, first_name, last_name,
    role, status, email_verified, staff_id
) VALUES (
    'manager',
    'manager@oasis.local',
    '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi',
    'John',
    'Manager',
    'Manager',
    'Active',
    1,
    1
);

-- Add tracking columns to existing tables
ALTER TABLE clients
ADD COLUMN created_by INT NULL AFTER created_at,
ADD COLUMN updated_by INT NULL AFTER updated_at,
ADD FOREIGN KEY (created_by) REFERENCES users(user_id) ON DELETE SET NULL,
ADD FOREIGN KEY (updated_by) REFERENCES users(user_id) ON DELETE SET NULL;

ALTER TABLE contact_records
ADD COLUMN created_by INT NULL AFTER created_at,
ADD FOREIGN KEY (created_by) REFERENCES users(user_id) ON DELETE SET NULL;

ALTER TABLE staff
ADD COLUMN created_by INT NULL AFTER created_at,
ADD COLUMN updated_by INT NULL AFTER updated_at,
ADD FOREIGN KEY (created_by) REFERENCES users(user_id) ON DELETE SET NULL,
ADD FOREIGN KEY (updated_by) REFERENCES users(user_id) ON DELETE SET NULL;

ALTER TABLE introducers
ADD COLUMN created_by INT NULL AFTER created_at,
ADD COLUMN updated_by INT NULL AFTER updated_at,
ADD FOREIGN KEY (created_by) REFERENCES users(user_id) ON DELETE SET NULL,
ADD FOREIGN KEY (updated_by) REFERENCES users(user_id) ON DELETE SET NULL;

-- Create view for user activity summary
CREATE OR REPLACE VIEW user_activity_summary AS
SELECT
    u.user_id,
    u.username,
    u.first_name,
    u.last_name,
    u.role,
    u.last_login,
    COUNT(DISTINCT al.log_id) as total_actions,
    COUNT(DISTINCT CASE WHEN al.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY) THEN al.log_id END) as actions_last_7_days,
    COUNT(DISTINCT CASE WHEN al.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN al.log_id END) as actions_last_30_days
FROM users u
LEFT JOIN activity_log al ON u.user_id = al.user_id
GROUP BY u.user_id, u.username, u.first_name, u.last_name, u.role, u.last_login;

-- Display results
SELECT 'Authentication system created successfully!' as Status;
SELECT CONCAT('Total Users: ', COUNT(*)) as Summary FROM users;
SELECT 'Default Admin Credentials:' as Note,
       'Username: admin' as Username,
       'Password: Admin@123' as Password,
       'CHANGE IMMEDIATELY!' as Warning;
```

### How to Run the SQL

**Option 1: Using phpMyAdmin**
1. Login to phpMyAdmin
2. Select `investor_db` database
3. Click "Import" tab
4. Choose file: `add_authentication.sql`
5. Click "Go"
6. Verify success message appears

**Option 2: Using MySQL Command Line**
```bash
mysql -u username -p investor_db < database/add_authentication.sql
```

**Option 3: Copy/Paste SQL**
1. Login to phpMyAdmin
2. Select `investor_db` database
3. Click "SQL" tab
4. Copy entire SQL script above
5. Paste into SQL box
6. Click "Go"

### Expected Results

After running SQL, you should see:
- ✅ 4 new tables created (users, user_sessions, activity_log, login_attempts)
- ✅ 2 default users created (admin, manager)
- ✅ 4 existing tables modified (clients, contact_records, staff, introducers)
- ✅ Confirmation message with default credentials

---

## Files to Create

You need to create **9 new files**. Each file is listed below with complete code.

### File List:
1. `database/add_authentication.sql` ✅ (created above)
2. `includes/auth.php` - Authentication handler
3. `login.php` - Login page
4. `logout.php` - Logout script
5. `access_denied.php` - Access denied page
6. `forms/user_management.php` - User management
7. `forms/change_password.php` - Password change
8. `forms/profile.php` - User profile
9. `forms/activity_log.php` - Activity log viewer

---

**[Continued in next message due to length...]**
