-- Authentication System for Investor Database
-- Version 2.2 - Security & Authentication
-- Run this after add_introducers.sql

USE investor_db;

-- Create Users Table
CREATE TABLE IF NOT EXISTS users (
    user_id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) UNIQUE NOT NULL,
    email VARCHAR(150) UNIQUE NOT NULL,
    password_hash VARCHAR(255) NOT NULL,

    -- Personal Information
    first_name VARCHAR(100) NOT NULL,
    last_name VARCHAR(100) NOT NULL,
    phone VARCHAR(20),

    -- User Role & Permissions
    role ENUM('Admin', 'Manager', 'User', 'Viewer') DEFAULT 'User',
    staff_id INT NULL,

    -- Account Status
    status ENUM('Active', 'Inactive', 'Suspended', 'Locked') DEFAULT 'Active',
    email_verified TINYINT(1) DEFAULT 0,

    -- Security
    failed_login_attempts INT DEFAULT 0,
    last_failed_login TIMESTAMP NULL,
    last_login TIMESTAMP NULL,
    last_password_change TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    password_reset_token VARCHAR(64) NULL,
    password_reset_expires TIMESTAMP NULL,

    -- Session Management
    remember_token VARCHAR(64) NULL,

    -- Timestamps
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    created_by INT NULL,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    updated_by INT NULL,

    FOREIGN KEY (staff_id) REFERENCES staff(staff_id) ON DELETE SET NULL,
    INDEX idx_username (username),
    INDEX idx_email (email),
    INDEX idx_role (role),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Create Sessions Table
CREATE TABLE IF NOT EXISTS user_sessions (
    session_id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    session_token VARCHAR(64) UNIQUE NOT NULL,
    ip_address VARCHAR(45),
    user_agent TEXT,
    expires_at TIMESTAMP NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    FOREIGN KEY (user_id) REFERENCES users(user_id) ON DELETE CASCADE,
    INDEX idx_session_token (session_token),
    INDEX idx_user_id (user_id),
    INDEX idx_expires (expires_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Create Activity Log Table
CREATE TABLE IF NOT EXISTS activity_log (
    log_id BIGINT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NULL,
    action VARCHAR(100) NOT NULL,
    entity_type VARCHAR(50) NULL,
    entity_id INT NULL,
    description TEXT,
    ip_address VARCHAR(45),
    user_agent TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    FOREIGN KEY (user_id) REFERENCES users(user_id) ON DELETE SET NULL,
    INDEX idx_user_id (user_id),
    INDEX idx_action (action),
    INDEX idx_entity (entity_type, entity_id),
    INDEX idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Create Login Attempts Table (for security monitoring)
CREATE TABLE IF NOT EXISTS login_attempts (
    attempt_id BIGINT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) NOT NULL,
    ip_address VARCHAR(45),
    success TINYINT(1) NOT NULL,
    failure_reason VARCHAR(100) NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    INDEX idx_username (username),
    INDEX idx_ip (ip_address),
    INDEX idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert default admin user
-- Default password: Admin@123 (MUST BE CHANGED IMMEDIATELY)
-- Password hash for: Admin@123
INSERT INTO users (
    username, email, password_hash, first_name, last_name,
    role, status, email_verified
) VALUES (
    'admin',
    'admin@oasis.local',
    '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi',
    'System',
    'Administrator',
    'Admin',
    'Active',
    1
);

-- Add a sample manager user
-- Default password: Manager@123 (MUST BE CHANGED)
INSERT INTO users (
    username, email, password_hash, first_name, last_name,
    role, status, email_verified, staff_id
) VALUES (
    'manager',
    'manager@oasis.local',
    '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi',
    'John',
    'Manager',
    'Manager',
    'Active',
    1,
    1
);

-- Add modified_by tracking to existing tables
ALTER TABLE clients
ADD COLUMN created_by INT NULL AFTER created_at,
ADD COLUMN updated_by INT NULL AFTER updated_at,
ADD FOREIGN KEY (created_by) REFERENCES users(user_id) ON DELETE SET NULL,
ADD FOREIGN KEY (updated_by) REFERENCES users(user_id) ON DELETE SET NULL;

ALTER TABLE contact_records
ADD COLUMN created_by INT NULL AFTER created_at,
ADD FOREIGN KEY (created_by) REFERENCES users(user_id) ON DELETE SET NULL;

ALTER TABLE staff
ADD COLUMN created_by INT NULL AFTER created_at,
ADD COLUMN updated_by INT NULL AFTER updated_at,
ADD FOREIGN KEY (created_by) REFERENCES users(user_id) ON DELETE SET NULL,
ADD FOREIGN KEY (updated_by) REFERENCES users(user_id) ON DELETE SET NULL;

ALTER TABLE introducers
ADD COLUMN created_by INT NULL AFTER created_at,
ADD COLUMN updated_by INT NULL AFTER updated_at,
ADD FOREIGN KEY (created_by) REFERENCES users(user_id) ON DELETE SET NULL,
ADD FOREIGN KEY (updated_by) REFERENCES users(user_id) ON DELETE SET NULL;

-- Create view for user activity summary
CREATE OR REPLACE VIEW user_activity_summary AS
SELECT
    u.user_id,
    u.username,
    u.first_name,
    u.last_name,
    u.role,
    u.last_login,
    COUNT(DISTINCT al.log_id) as total_actions,
    COUNT(DISTINCT CASE WHEN al.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY) THEN al.log_id END) as actions_last_7_days,
    COUNT(DISTINCT CASE WHEN al.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN al.log_id END) as actions_last_30_days
FROM users u
LEFT JOIN activity_log al ON u.user_id = al.user_id
GROUP BY u.user_id, u.username, u.first_name, u.last_name, u.role, u.last_login;

-- Display results
SELECT 'Authentication system created successfully!' as Status;
SELECT CONCAT('Total Users: ', COUNT(*)) as Summary FROM users;
SELECT 'Default Admin Credentials:' as Note, 'Username: admin' as Username, 'Password: Admin@123' as Password, 'CHANGE IMMEDIATELY!' as Warning;
