-- Add Introducers Table and Update Clients Table
-- Run this after schema.sql to add introducer functionality

USE investor_db;

-- Create Introducers Table
CREATE TABLE IF NOT EXISTS introducers (
    introducer_id INT AUTO_INCREMENT PRIMARY KEY,

    -- Contact Information
    company_name VARCHAR(200),
    contact_person VARCHAR(150) NOT NULL,
    first_name VARCHAR(100),
    last_name VARCHAR(100),
    email VARCHAR(150) UNIQUE NOT NULL,
    phone VARCHAR(20),
    mobile VARCHAR(20),
    address_line1 VARCHAR(200),
    address_line2 VARCHAR(200),
    city VARCHAR(100),
    state_province VARCHAR(100),
    postal_code VARCHAR(20),
    country VARCHAR(100),
    website VARCHAR(200),

    -- KYC Due Diligence
    kyc_status ENUM('Pending', 'In Progress', 'Completed', 'Expired') DEFAULT 'Pending',
    kyc_completion_date DATE,
    kyc_expiry_date DATE,
    business_registration_number VARCHAR(100),
    tax_id VARCHAR(50),
    regulatory_license VARCHAR(100),
    license_expiry_date DATE,
    pep_status ENUM('Yes', 'No', 'Unknown') DEFAULT 'Unknown',
    risk_rating ENUM('Low', 'Medium', 'High') DEFAULT 'Medium',

    -- Business Information
    business_type VARCHAR(100),
    years_in_business INT,
    number_of_clients INT,
    commission_rate DECIMAL(5,2),
    commission_structure TEXT,

    -- Additional Information
    notes TEXT,
    introducer_status ENUM('Active', 'Inactive', 'Suspended') DEFAULT 'Active',
    assigned_staff_id INT,

    -- Timestamps
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    last_contact_date TIMESTAMP NULL,

    FOREIGN KEY (assigned_staff_id) REFERENCES staff(staff_id) ON DELETE SET NULL,
    INDEX idx_company_name (company_name),
    INDEX idx_email (email),
    INDEX idx_status (introducer_status),
    INDEX idx_kyc_status (kyc_status),
    INDEX idx_assigned_staff (assigned_staff_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Add introducer_id and owner_staff_id to clients table
ALTER TABLE clients
ADD COLUMN introducer_id INT AFTER assigned_staff_id,
ADD COLUMN owner_staff_id INT AFTER introducer_id,
ADD FOREIGN KEY (introducer_id) REFERENCES introducers(introducer_id) ON DELETE SET NULL,
ADD FOREIGN KEY (owner_staff_id) REFERENCES staff(staff_id) ON DELETE SET NULL,
ADD INDEX idx_introducer (introducer_id),
ADD INDEX idx_owner_staff (owner_staff_id);

-- Insert sample introducers
INSERT INTO introducers (
    company_name, contact_person, first_name, last_name, email, phone,
    city, country, kyc_status, risk_rating, business_type,
    years_in_business, commission_rate, introducer_status, assigned_staff_id
) VALUES
(
    'Premier Wealth Advisors', 'John Sterling', 'John', 'Sterling',
    'john.sterling@premierwealth.com', '+1-555-2001',
    'New York', 'United States', 'Completed', 'Low',
    'Independent Financial Advisor', 15, 1.50, 'Active', 1
),
(
    'Global Investment Partners', 'Sarah Chen', 'Sarah', 'Chen',
    'sarah.chen@globalinvest.com', '+44-20-5552001',
    'London', 'United Kingdom', 'Completed', 'Low',
    'Investment Consultancy', 8, 2.00, 'Active', 2
),
(
    'Elite Financial Services', 'Michael Rodriguez', 'Michael', 'Rodriguez',
    'michael.r@elitefinancial.com', '+1-555-3001',
    'Miami', 'United States', 'In Progress', 'Medium',
    'Financial Planning Firm', 5, 1.75, 'Active', 1
);

-- Update some existing clients with introducers and owners
UPDATE clients SET introducer_id = 1, owner_staff_id = 1 WHERE client_id = 1;
UPDATE clients SET introducer_id = 2, owner_staff_id = 2 WHERE client_id = 2;
UPDATE clients SET owner_staff_id = 3 WHERE client_id = 3;

-- Display results
SELECT 'Introducers table created successfully!' as Status;
SELECT CONCAT('Total Introducers: ', COUNT(*)) as Summary FROM introducers;
