<?php
// Authentication and Session Management
require_once 'config.php';
require_once 'db.php';

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Check if user is logged in
function is_logged_in() {
    return isset($_SESSION['user_id']) && isset($_SESSION['username']);
}

// Require login - redirect to login page if not authenticated
function require_login() {
    if (!is_logged_in()) {
        $redirect_url = urlencode($_SERVER['REQUEST_URI']);
        header("Location: " . get_base_path() . "login.php?redirect=" . $redirect_url);
        exit();
    }
}

// Check if user has required role
function has_role($required_role) {
    if (!is_logged_in()) {
        return false;
    }

    $user_role = $_SESSION['role'] ?? 'User';

    $role_hierarchy = [
        'Admin' => 4,
        'Manager' => 3,
        'User' => 2,
        'Viewer' => 1
    ];

    $user_level = $role_hierarchy[$user_role] ?? 0;
    $required_level = $role_hierarchy[$required_role] ?? 0;

    return $user_level >= $required_level;
}

// Require specific role
function require_role($required_role) {
    require_login();

    if (!has_role($required_role)) {
        header("Location: " . get_base_path() . "access_denied.php");
        exit();
    }
}

// Get base path
function get_base_path() {
    $current_dir = dirname($_SERVER['PHP_SELF']);
    return (strpos($current_dir, '/forms') !== false) ? '../' : '';
}

// Get authenticated user data
function get_authenticated_user() {
    global $conn;

    if (!is_logged_in()) {
        return null;
    }

    $user_id = intval($_SESSION['user_id']);
    $query = "SELECT * FROM users WHERE user_id = $user_id AND status = 'Active'";
    $result = $conn->query($query);

    if ($result && $result->num_rows > 0) {
        return $result->fetch_assoc();
    }

    return null;
}

// Login user
function login_user($username, $password, $remember = false) {
    global $conn;

    $username = sanitize_input($username);
    $ip_address = $_SERVER['REMOTE_ADDR'];

    // Get user from database
    $query = "SELECT * FROM users WHERE username = '$username' AND status IN ('Active')";
    $result = $conn->query($query);

    if ($result && $result->num_rows > 0) {
        $user = $result->fetch_assoc();

        // Check if account is locked (5 failed attempts in last 15 minutes)
        if ($user['failed_login_attempts'] >= 5) {
            $last_failed = strtotime($user['last_failed_login']);
            $lockout_time = 900; // 15 minutes

            if (time() - $last_failed < $lockout_time) {
                log_login_attempt($username, $ip_address, false, 'Account locked');
                return ['success' => false, 'message' => 'Account locked. Please try again in 15 minutes.'];
            } else {
                // Reset failed attempts after lockout period
                $conn->query("UPDATE users SET failed_login_attempts = 0 WHERE user_id = {$user['user_id']}");
            }
        }

        // Verify password
        if (password_verify($password, $user['password_hash'])) {
            // Successful login
            $_SESSION['user_id'] = $user['user_id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['role'] = $user['role'];
            $_SESSION['first_name'] = $user['first_name'];
            $_SESSION['last_name'] = $user['last_name'];
            $_SESSION['email'] = $user['email'];

            // Update last login and reset failed attempts
            $conn->query("UPDATE users SET
                last_login = NOW(),
                failed_login_attempts = 0,
                last_failed_login = NULL
                WHERE user_id = {$user['user_id']}");

            // Create session token
            $session_token = bin2hex(random_bytes(32));
            $expires_at = date('Y-m-d H:i:s', time() + 86400); // 24 hours
            $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';

            $conn->query("INSERT INTO user_sessions (user_id, session_token, ip_address, user_agent, expires_at)
                         VALUES ({$user['user_id']}, '$session_token', '$ip_address', '$user_agent', '$expires_at')");

            $_SESSION['session_token'] = $session_token;

            // Handle remember me
            if ($remember) {
                $remember_token = bin2hex(random_bytes(32));
                setcookie('remember_token', $remember_token, time() + 2592000, '/'); // 30 days
                $conn->query("UPDATE users SET remember_token = '$remember_token' WHERE user_id = {$user['user_id']}");
            }

            // Log successful login
            log_login_attempt($username, $ip_address, true, null);
            log_activity($user['user_id'], 'login', null, null, 'User logged in');

            return ['success' => true, 'user' => $user];
        } else {
            // Failed login - increment failed attempts
            $failed_attempts = $user['failed_login_attempts'] + 1;
            $conn->query("UPDATE users SET
                failed_login_attempts = $failed_attempts,
                last_failed_login = NOW()
                WHERE user_id = {$user['user_id']}");

            log_login_attempt($username, $ip_address, false, 'Invalid password');
            return ['success' => false, 'message' => 'Invalid username or password.'];
        }
    } else {
        // User not found
        log_login_attempt($username, $ip_address, false, 'User not found');
        return ['success' => false, 'message' => 'Invalid username or password.'];
    }
}

// Logout user
function logout_user() {
    global $conn;

    if (isset($_SESSION['user_id'])) {
        $user_id = $_SESSION['user_id'];

        // Delete session from database
        if (isset($_SESSION['session_token'])) {
            $session_token = $_SESSION['session_token'];
            $conn->query("DELETE FROM user_sessions WHERE session_token = '$session_token'");
        }

        // Log logout
        log_activity($user_id, 'logout', null, null, 'User logged out');
    }

    // Clear session
    $_SESSION = array();

    // Delete session cookie
    if (isset($_COOKIE[session_name()])) {
        setcookie(session_name(), '', time() - 3600, '/');
    }

    // Delete remember me cookie
    if (isset($_COOKIE['remember_token'])) {
        setcookie('remember_token', '', time() - 3600, '/');
    }

    // Destroy session
    session_destroy();
}

// Log login attempt
function log_login_attempt($username, $ip_address, $success, $failure_reason = null) {
    global $conn;

    $username = sanitize_input($username);
    $ip_address = sanitize_input($ip_address);
    $success = $success ? 1 : 0;
    $failure_reason = $failure_reason ? "'" . sanitize_input($failure_reason) . "'" : "NULL";

    $conn->query("INSERT INTO login_attempts (username, ip_address, success, failure_reason)
                 VALUES ('$username', '$ip_address', $success, $failure_reason)");
}

// Log activity
function log_activity($user_id, $action, $entity_type = null, $entity_id = null, $description = null) {
    global $conn;

    $user_id = intval($user_id);
    $action = sanitize_input($action);
    $entity_type = $entity_type ? "'" . sanitize_input($entity_type) . "'" : "NULL";
    $entity_id = $entity_id ? intval($entity_id) : "NULL";
    $description = $description ? "'" . sanitize_input($description) . "'" : "NULL";
    $ip_address = $_SERVER['REMOTE_ADDR'];
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';

    $conn->query("INSERT INTO activity_log (user_id, action, entity_type, entity_id, description, ip_address, user_agent)
                 VALUES ($user_id, '$action', $entity_type, $entity_id, $description, '$ip_address', '$user_agent')");
}

// Generate CSRF token
function generate_csrf_token() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

// Verify CSRF token
function verify_csrf_token($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

// Clean expired sessions (call periodically)
function clean_expired_sessions() {
    global $conn;
    $conn->query("DELETE FROM user_sessions WHERE expires_at < NOW()");
}

// Auto-login with remember token
function auto_login_remember_token() {
    global $conn;

    if (isset($_COOKIE['remember_token']) && !is_logged_in()) {
        $remember_token = sanitize_input($_COOKIE['remember_token']);

        $query = "SELECT * FROM users WHERE remember_token = '$remember_token' AND status = 'Active'";
        $result = $conn->query($query);

        if ($result && $result->num_rows > 0) {
            $user = $result->fetch_assoc();

            $_SESSION['user_id'] = $user['user_id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['role'] = $user['role'];
            $_SESSION['first_name'] = $user['first_name'];
            $_SESSION['last_name'] = $user['last_name'];
            $_SESSION['email'] = $user['email'];

            // Update last login
            $conn->query("UPDATE users SET last_login = NOW() WHERE user_id = {$user['user_id']}");

            // Create new session token
            $session_token = bin2hex(random_bytes(32));
            $expires_at = date('Y-m-d H:i:s', time() + 86400);
            $ip_address = $_SERVER['REMOTE_ADDR'];
            $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';

            $conn->query("INSERT INTO user_sessions (user_id, session_token, ip_address, user_agent, expires_at)
                         VALUES ({$user['user_id']}, '$session_token', '$ip_address', '$user_agent', '$expires_at')");

            $_SESSION['session_token'] = $session_token;

            log_activity($user['user_id'], 'auto_login', null, null, 'Auto-login via remember token');
        } else {
            // Invalid token, delete cookie
            setcookie('remember_token', '', time() - 3600, '/');
        }
    }
}

// Try auto-login on every page load
auto_login_remember_token();

// Clean expired sessions occasionally (1% chance)
if (rand(1, 100) === 1) {
    clean_expired_sessions();
}
?>
