<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Database Connection Test</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            max-width: 800px;
            margin: 50px auto;
            padding: 20px;
            background-color: #f4f4f4;
        }
        .container {
            background: white;
            padding: 30px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #2c3e50;
            border-bottom: 3px solid #3498db;
            padding-bottom: 10px;
        }
        .success {
            background-color: #d4edda;
            color: #155724;
            padding: 15px;
            border-radius: 4px;
            border: 1px solid #c3e6cb;
            margin: 20px 0;
        }
        .error {
            background-color: #f8d7da;
            color: #721c24;
            padding: 15px;
            border-radius: 4px;
            border: 1px solid #f5c6cb;
            margin: 20px 0;
        }
        .info {
            background-color: #d1ecf1;
            color: #0c5460;
            padding: 15px;
            border-radius: 4px;
            border: 1px solid #bee5eb;
            margin: 20px 0;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
        }
        th, td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #ddd;
        }
        th {
            background-color: #2c3e50;
            color: white;
        }
        .btn {
            display: inline-block;
            padding: 10px 20px;
            background-color: #3498db;
            color: white;
            text-decoration: none;
            border-radius: 4px;
            margin-top: 20px;
        }
        .btn:hover {
            background-color: #2980b9;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>Database Connection Test</h1>

        <?php
        require_once 'includes/config.php';

        $all_tests_passed = true;

        // Test 1: Check if config is loaded
        echo "<h2>Configuration Check</h2>";
        if (defined('DB_HOST') && defined('DB_USER') && defined('DB_NAME')) {
            echo "<div class='success'>✓ Configuration file loaded successfully</div>";
            echo "<table>";
            echo "<tr><th>Setting</th><th>Value</th></tr>";
            echo "<tr><td>Database Host</td><td>" . DB_HOST . "</td></tr>";
            echo "<tr><td>Database User</td><td>" . DB_USER . "</td></tr>";
            echo "<tr><td>Database Name</td><td>" . DB_NAME . "</td></tr>";
            echo "<tr><td>PHP Version</td><td>" . phpversion() . "</td></tr>";
            echo "</table>";
        } else {
            echo "<div class='error'>✗ Configuration file not loaded properly</div>";
            $all_tests_passed = false;
        }

        // Test 2: Database Connection
        echo "<h2>Database Connection Test</h2>";
        $conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);

        if ($conn->connect_error) {
            echo "<div class='error'>✗ Connection Failed: " . $conn->connect_error . "</div>";
            echo "<div class='info'>";
            echo "<strong>Troubleshooting:</strong><br>";
            echo "1. Check if MySQL/MariaDB is running<br>";
            echo "2. Verify database credentials in includes/config.php<br>";
            echo "3. Ensure the database 'investor_db' exists<br>";
            echo "4. Check database user permissions";
            echo "</div>";
            $all_tests_passed = false;
        } else {
            echo "<div class='success'>✓ Database connection successful!</div>";

            // Test 3: Check Tables
            echo "<h2>Database Tables Check</h2>";
            $tables_to_check = ['staff', 'clients', 'contact_records'];
            $tables_exist = true;

            echo "<table>";
            echo "<tr><th>Table Name</th><th>Status</th><th>Record Count</th></tr>";

            foreach ($tables_to_check as $table) {
                $result = $conn->query("SHOW TABLES LIKE '$table'");
                if ($result && $result->num_rows > 0) {
                    $count_result = $conn->query("SELECT COUNT(*) as count FROM $table");
                    $count = $count_result->fetch_assoc()['count'];
                    echo "<tr>";
                    echo "<td>$table</td>";
                    echo "<td style='color: green;'>✓ Exists</td>";
                    echo "<td>$count records</td>";
                    echo "</tr>";
                } else {
                    echo "<tr>";
                    echo "<td>$table</td>";
                    echo "<td style='color: red;'>✗ Missing</td>";
                    echo "<td>N/A</td>";
                    echo "</tr>";
                    $tables_exist = false;
                }
            }
            echo "</table>";

            if (!$tables_exist) {
                echo "<div class='error'>✗ Some tables are missing. Please import database/schema.sql</div>";
                $all_tests_passed = false;
            } else {
                echo "<div class='success'>✓ All required tables exist!</div>";
            }

            // Test 4: Sample Data Check
            echo "<h2>Sample Data Check</h2>";
            $staff_count = $conn->query("SELECT COUNT(*) as count FROM staff")->fetch_assoc()['count'];

            if ($staff_count > 0) {
                echo "<div class='success'>✓ Found $staff_count staff members in database</div>";

                echo "<h3>Staff Members:</h3>";
                $staff_result = $conn->query("SELECT first_name, last_name, position, department FROM staff ORDER BY last_name");
                echo "<table>";
                echo "<tr><th>Name</th><th>Position</th><th>Department</th></tr>";
                while ($row = $staff_result->fetch_assoc()) {
                    echo "<tr>";
                    echo "<td>{$row['first_name']} {$row['last_name']}</td>";
                    echo "<td>{$row['position']}</td>";
                    echo "<td>{$row['department']}</td>";
                    echo "</tr>";
                }
                echo "</table>";
            } else {
                echo "<div class='info'>ℹ No staff members found. The schema imported correctly but no sample data was loaded.</div>";
            }

            $conn->close();
        }

        // Final Summary
        echo "<h2>Test Summary</h2>";
        if ($all_tests_passed) {
            echo "<div class='success'>";
            echo "<h3>✓ All Tests Passed!</h3>";
            echo "<p>Your database is configured correctly and ready to use.</p>";
            echo "<a href='index.php' class='btn'>Go to Dashboard</a>";
            echo "</div>";
        } else {
            echo "<div class='error'>";
            echo "<h3>✗ Some Tests Failed</h3>";
            echo "<p>Please review the errors above and fix the issues.</p>";
            echo "<p>Refer to SETUP_INSTRUCTIONS.txt for help.</p>";
            echo "</div>";
        }
        ?>

        <div class="info" style="margin-top: 30px;">
            <strong>Note:</strong> For security reasons, delete or restrict access to this file (test_connection.php)
            after confirming your setup is working correctly.
        </div>
    </div>
</body>
</html>
